/******************** (C) COPYRIGHT 2007 STMicroelectronics ********************
* File Name          : EEPROM.c
* Author             : MCD APPLICATION TEAM
* Date First Issued  : 05/30/2007 : Version 1.0
* Description        : EEPROM emulation routines
********************************************************************************
* History:
* 05/30/2007 : Version 1.0
********************************************************************************
THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME. AS A
RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR
CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT OF SUCH
SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION CONTAINED
HEREININ CONNECTION WITH THEIR PRODUCTS.
********************************************************************************/


/* Standard include ----------------------------------------------------------*/
#include "91x_lib.h"
#include "EEPROM.h"

/* Include of other module interface headers ---------------------------------*/
/* Local includes ------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/

/* Private functions ---------------------------------------------------------*/
u8 FindValidSector(u8 operation);
u16 WriteVerifyVariableFull(u8 index, u32 *T, u16 Data);
u8 EepromSectorTransfer(u8 index, u32 *T, u16 data);
/*******************************************************************************
* Routine :  EepromFormat
* Purpose :  Format the whole EEPROM
* Input   :
* Output  : Status of the operation
*******************************************************************************/

u8 EepromFormat(void)
{
  u8 Status;
  u8 FMI_Timeout_Status1 ;
  u8 FMI_Timeout_Status2 ;
  FMI_EraseSector(FMI_B1S0);

  /* Wait until erase operation is complete */
  FMI_Timeout_Status1 = FMI_WaitForLastOperation(FMI_BANK_1);


  if (FMI_Timeout_Status1 == 1)
  {
    /* Declare Sector0 as VALID_SECTOR:write VALID_SECTOR at sector0 baseaddress*/
    FMI_WriteHalfWord( FMI_B1S0, 0xCCCC);
    FMI_Timeout_Status2 = FMI_WaitForLastOperation(FMI_BANK_1);

    /* if program operation was failed, a Flash error code is returned*/
    if (FMI_Timeout_Status2 != 1)
      Status = FMI_Timeout_Status2;
  }
  else
  {
    /*if erase operation is not successful so return Flash error code */
    Status = FMI_Timeout_Status1;
  }
  /* Erase sector1 */
  FMI_EraseSector(FMI_B1S1);
  /* Wait until erase operation is complete */
  FMI_Timeout_Status1 = FMI_WaitForLastOperation(FMI_BANK_1);

  Status = FMI_Timeout_Status1;
  return  Status;

}

/*******************************************************************************
* Routine:  FindValidSector
* Purpose:  Find valid sector for write or read operation
* Input  :  a byte indicating whether we search valid sector for read from or
*           write into( READ_FROM_VALID_SECTOR or WRITE_IN_VALID_SECTOR
* Output:   Sector number
*******************************************************************************/

u8 FindValidSector(u8 operation)
{
  u16  STATUS0, STATUS1;
  /* Check sector 0 for status */
  STATUS0 = FMI_ReadWord(FMI_B1S0);

  /* Check sector 1 for status */
  STATUS1 = FMI_ReadWord(FMI_B1S1);
  /*write or read: either we want to write in or read from the suitable sector*/
  switch (operation)
  {

    case WRITE_IN_VALID_SECTOR:
      if ( STATUS1 == VALID_SECTOR)
      {
        /* sector0 receiving data */
        if ( STATUS0 == RECEIVE_DATA)  return (SECTOR0);
        else return (SECTOR1);
      }
      else if ( STATUS0 == VALID_SECTOR)
      {
        /* sector1 receiving data */
        if ( STATUS1 == RECEIVE_DATA) return (SECTOR1);
        else return (SECTOR0);
      }
      else return No_Valid_Sector;
    case READ_FROM_VALID_SECTOR:

      if ( STATUS0 == VALID_SECTOR)   return (SECTOR0);

      else if ( STATUS1 == VALID_SECTOR)  return (SECTOR1);
      else return No_Valid_Sector ;

    default:
      return (SECTOR0);
  }
}

/*******************************************************************************
* Routine:  WriteVerifyVariableFull
* Purpose:  Writes  variable in EEPROM.
* Input  :  - index: variable identifier(0 for A, 1 for B, 2 for C
*           - T: variables Array
*           - Data: 16 bit data to be written
* Output:   Returns 1 on success or 0x80 if variable is full or Flash error code
*******************************************************************************/

u16 WriteVerifyVariableFull(u8 index, u32 *T, u16 Data)

{

  u8 valid_sector;
  s16 status, status1, status2, status3 = -1;
  u32 address, variable_end_address;
  /* Get valid sector for write operation */
  valid_sector = FindValidSector(WRITE_IN_VALID_SECTOR);
  if (valid_sector == No_Valid_Sector) return  No_Valid_Sector;
  /* Get the variable start address in the valid sector */
  address = (u32)(T[index] + (u32)(valid_sector * SECTOR_SIZE));

  /* Get the variable end address in the valid sector */

  /* If the variable isn't the last one in the sector */
  if (index != max_varia - 1)
  {
    variable_end_address = (u32)(T[index+1]) ;
    variable_end_address += (u32)(valid_sector * SECTOR_SIZE);
  }
  else
    /*variable is the last one in the sector, it occupies the remaining space
    memory in the sector: until SECTOR_END_ADDRESS)*/
  {
    variable_end_address = (u32)SECTOR0_END_ADDRESS;
    variable_end_address += (u32)(valid_sector  * SECTOR_SIZE);
  }
  do
  {  /* verify if address content is 0xFFFF ; if yes status = 0*/
    if ((u16)(FMI_ReadWord(address)) == 0xFFFF) status = 0 ;
    else  status =0xBADC;

    /* verify if (address + 2) content is 0xFFFF ; if yes status1 = 0*/

    if ((u16)(FMI_ReadWord(address + 2)) == 0xFFFF) status1 = 0 ;
    else  status1 =0xBADC;
    /* verify if at (address + 4) content is 0x0000; if yes status2 = 0 */

    if ((u16)(FMI_ReadWord(address + 2)) == 0x0000) status2 = 0 ;
   else  status2 =0xBADC;

    if ((status == 0) && (status1 == 0))
    {
      if (Data != 0xFFFF)
      {
        /* address is a blank location so write the data */

        FMI_WriteHalfWord( address, Data);
        status3 = FMI_WaitForLastOperation(FMI_BANK_1);
        break;
      }
      else /* Data = 0xFFFF */
      {
        address = address + 2;
        if (address >= variable_end_address)
        {
          status3 = VARIABLE_FULL;
          break;
        }
        /* Write 0x0000 at address + 2 */

        FMI_WriteHalfWord( address, 0x0000);
        status3 = FMI_WaitForLastOperation(FMI_BANK_1);
        break;
      }
    }
    /* 0xFFFF is a data */
    else if ((status == 0) && (status2 == 0))  address = address + 4;
    else address = address + 2;
    if (address >= (variable_end_address - 2))
    {
      status3 = VARIABLE_FULL; /* 0x80 is returned */
      break;
    }
  }
  while (address < (variable_end_address - 2));
  return status3;

}
/******************************************************************************
*Routine:  ReadVariable
*Purpose:  Reads the last stored variable data
*Input  : - index: variable identifier
*          - T: variable array
* Output:  Returns 16 bit read data on success or error code on failure
*******************************************************************************/

u16 ReadVariable(u8 index, u32 *T)
{
  u8 valid_sector;
  u32 address, base_address, variable_end_address;
  u16 Read_data;
  /* Get active sector  for READ operation */
  valid_sector = FindValidSector(READ_FROM_VALID_SECTOR);
  if (valid_sector == No_Valid_Sector) return  No_Valid_Sector;
  /* Get the last location in the space memory allocated for the variable */
  if (index != max_varia - 1) /* If variable isn't the last one in the sector */
  {
    variable_end_address = T[index + 1] + (valid_sector  * SECTOR_SIZE);
  }
  else  /* If variable is the last one in the sector */
  {
    variable_end_address = SECTOR0_END_ADDRESS;
    variable_end_address += valid_sector  * SECTOR_SIZE;
  }

  /* Get variable base address in the valid sector  */
  base_address = (u32)(T[index] + (valid_sector  * SECTOR_SIZE));
  address = base_address;
  do
  {
    /* Go through the variable beginning with its base address until
    finding a blanck location,then return the data in the previous location */
    /* Read address content */
    Read_data =  FMI_ReadWord(address);
    if (Read_data != 0xFFFF)    /* If location is not blank */
    {
      address = address + 2;    /* Increment + 2 for second location */
    }
    else /* Read_data = 0xFFFF */
    {
      /*0xFFFF is a data and address is not a blank location*/
      if ((u16)FMI_ReadWord(address + 2) == 0x0000)
      {
        /* Increment +4: 2 for 0xFFFF and 2 for 0x0000 */
        address = address + 4;
      }
      else if ((u16)FMI_ReadWord(address + 2) == 0xFFFF)
      {
        if (address != base_address)
        {
          /* case of last update = 0xFFFF */
          if (((u16)FMI_ReadWord(address - 4) == 0xFFFF) &&
              ((u16)FMI_ReadWord(address - 2) == 0x0000))

          {
            Read_data = 0xFFFF;
            break;
          }
          else
          {
            Read_data =  (u16)FMI_ReadWord(address - 2);
            break;
          }
        }
        else /* address = variable base address so the variable is EMPTY */
        {
          Read_data = VARIABLE_EMPTY;
          break;
        }
      }
    }
  }
  while (address < (variable_end_address - 2));

  return Read_data;

}


/*******************************************************************************
* Routine:  EepromSectorTransfer
* Purpose:  Transfers data from full sector to an empty one.
* Input  :  - index:variable identifier(0, 1, 2 ...)
*           - T: Variables base Array
*           - data: 16 bit data to be written
* Output:   returns 1 on success or error code
*******************************************************************************/

u8 EepromSectorTransfer(u8 index, u32 *T, u16 data)
{
  u8  valid_sector, status = 0;
  u16 Data;
  u32  new_sector, old_sector, j = 0;
  /* new_sector : The sector we are moving to */
  /* old_sector : The sector we are moving from*/

  /* Get active sector  (This is the sector we are moving from) */
  valid_sector = FindValidSector(READ_FROM_VALID_SECTOR);
  if (valid_sector == SECTOR1)
  {
    new_sector = SECTOR0_BASE_ADDRESS;  /* move data to this sector*/
    old_sector = SECTOR1_BASE_ADDRESS;  /* move data from this sector*/

  }
  else if (valid_sector == SECTOR0)
  {
    new_sector = SECTOR1_BASE_ADDRESS; /* move data to this sector*/
    old_sector = SECTOR0_BASE_ADDRESS; /* move data from this sector*/

  }
  else return  No_Valid_Sector;

  /* Mark new sector to receive */
  FMI_WriteHalfWord( new_sector, RECEIVE_DATA);
  status = FMI_WaitForLastOperation(FMI_BANK_1);


  /* if error, return error code */
  if (status != 1) return status;
  /* Transfer process */
  do
  {
    /* Write the variable data */
    if (j == index)
    {
      status = WriteVerifyVariableFull(index, T, data);
      if (status != 1)
        return status;
    }
    /* Transfer other variables last updates */
    else
    {
      /* Read the other last variable updates */
      Data = ReadVariable(j, T);
      if (Data != VARIABLE_EMPTY)
        status = WriteVerifyVariableFull(j, T, Data);
      if (status != 1)
        return status;
    }
    j++;
  }
  while (j < max_varia);
  /* Mark good and bad sectors */
  /* Mark old sector TRANSFER_COMPLETE before marking new sector VALID_SECTOR */

  FMI_WriteHalfWord( old_sector, TRANSFER_COMPLETE);
  status = FMI_WaitForLastOperation(FMI_BANK_1);
  if (status != 1) return status;
  /* mark new sector VALID_SECTOR after marking old sector TRANSFER_COMPLETE*/
  FMI_WriteHalfWord( new_sector, VALID_SECTOR);
  status = FMI_WaitForLastOperation(FMI_BANK_1);
  if (status != 1) return status;
  /* Erase the old sector */
  FMI_EraseSector(old_sector);
  /* Wait until erase operation is complete */
  status = FMI_WaitForLastOperation(FMI_BANK_1);

  return status;
}

/*******************************************************************************
* Routine:  WriteVariable
* Purpose:  Writes/upadtes  variable data in EEPROM.
* Input  :  - index: variable identifier
*           - T: variables array
*           - data: 16 bit data to be written
* Output:   returns 1 on success or error code on failure
*******************************************************************************/
u8 WriteVariable(u8 index, u32 *T, u16 data)
{
  u16 status;
  status = WriteVerifyVariableFull(index, T, data);
  /* When there isn't enough memory space for the new variable value, the new
  datavariable + the last variables' updates are transferred to a blank sector*/
  if (status == VARIABLE_FULL)
  {
    /* Perform sector transfer*/
    status = EepromSectorTransfer(index, T, data);
  }
  return status;
}

/******************* (c) 2007  ST Microelectronics *********** END OF FILE ****/
